 /**
 *
 * \file        HPRFGW_RFDesc.c
 *
 * \brief       This file contains the RF descriptor table creation and other related routines
 *
 * \detail
 *
 * \author      Hazrat Shah
 *
 * \date        11/29/2006
 * \warning
 * \note
 * \review
 */

#include "hprfgw_config.h"
#include "hprfgw_rfSlaveIntToHost.h"
#include "hprfgw_rfHeartbeat.h"
#include "hprfgw_rfDesc.h"
#include "hprfgw_rfTrans.h"
#include "hprfgw_rfSharedDefs.h"
#include "hprfgw_rfSlaveIntToRfApp.h"
#include "hprfgw_rfSlaveIntToHost.h"


TYPE_RF_DEVICE_BROADCAST_SEQ_REC BDR;

////////////////////////////////////////////////////////////////////////////////
//	DEFINITIONS
////////////////////////////////////////////////////////////////////////////////

//#define	RF_DEVICE_PIPE_SIZE_LOW_TRAFFIC_NODE	(1024*2)
//#define	RF_DEVICE_PIPE_SIZE_MED_TRAFFIC_NODE	(1024*5)
//#define	RF_DEVICE_PIPE_SIZE_HIGH_TRAFFIC_NODE	(1024*20)		

#define		GATEWAY_MEMORY_SIZE_FOR_ALL_PIPES			(1024*3000)		//	total pipe memory in gateway
#define		GATEWAY_MAX_ALLOWABLE_MEMORY_PER_PIPE		(1024*200)		//	max pipe size per device in gateway

#define		SLAVE_DEVICE_MEMORY_SIZE_FOR_ALL_PIPES		(1024*300)		//	total pipe memory in slave device
#define		SLAVE_DEVICE_MAX_ALLOWABLE_MEMORY_PER_PIPE	(1024*10)		//	max pipe size per device in slave device

//	RF Device table index semaphore timeout		
#if defined (OS_NUCLEUS)
 #define RFDEVBYRFID_SEMAPHORE_TIMEOUT			(TICKS_PER_SECOND)	//	1-second 
#elif defined (OS_WINCE)
  #define RFDEVBYRFID_SEMAPHORE_TIMEOUT			(1000)				//	1-second, should be in mS.
#endif
#define RFDEVBYRFID_SEMAPHORE_TIMEOUT_MS			(1000)

////////////////////////////////////////////////////////////////////////////////
//	LOCAL DATA
////////////////////////////////////////////////////////////////////////////////
typedef	struct	{									
	UINT8	index[MAX_DESCRIPTIVE_TABLE_ENTRIES];
	UINT8	count;
}	TYPE_RF_DEVICE_TABLE_BYRFID;
TYPE_RF_DEVICE_TABLE_BYRFID RFDevByRFID[MAX_DESCRIPTIVE_TABLE_ENTRIES];
#if defined (OS_NUCLEUS)
 NU_SEMAPHORE RFDevByRFID_Semaphore;
#elif defined (OS_WINCE)
 HANDLE RFDevByRFID_Semaphore;
#endif
DM_RESOURCE_HANDLE RFDevByRFID_Semaphore;

UINT8	LinkQualityHesterisas;
UINT8	g_MaxSupportedRFDevices;
UINT8	g_MaxDescriptiveTableEntries;
TYPE_RF_DESCRIPTIVE_TABLE* g_pRFDeviceDescriptiveTable;
BOOL	AcquireMode;
BOOL	RFDescEntriesCreated = FALSE;
UINT8   g_roamingEnabled = 0;
#ifndef	RF_NETWORK_MASTER_DEVICE
TYPE_RF_DESCRIPTIVE_TABLE* g_pRFRoamingDescriptiveTable = NULL;
INT8    g_acquireSlot = 1;//slot is from 1 to 8, table entry index is 0 to 7
#endif

// MNT - 3/20/2007 - The app should define this
#if defined (OS_NUCLEUS)
 extern NU_MEMORY_POOL  System_Memory;
#endif

#ifdef	RF_NETWORK_MASTER_DEVICE
UINT16 EEpromGetMaxSupportedRFDeviceCount (UINT8 *pMaxSlaveRFDevices);
#include "hprfgw_cresnet.h"
#endif //#ifdef	RF_NETWORK_MASTER_DEVICE

TYPE_RF_MSTAT_PARS gMStatp;

////////////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
//	LOCAL FUNCTIONS
///////////////////////////////////////////////////////////////////////////////

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This routines returns link state of device of given RFID
 *
 * \detail
 * \date        08/09/2007
 *
 * \param	    UINT8 rfid, UINT8 *plinkstate
 *
 * \return		UINT16 res
 *
 * \retval		SUCCESS/FAILURE
 */
UINT16	RFDesc_GetDeviceLinkStateByRFID (UINT8 rfid, UINT8 *plinkstate) {
	UINT16	i, res = FAILURE;
	UINT8	index[MAX_DESCRIPTIVE_TABLE_ENTRIES];

	*plinkstate = RF_DEVICE_LINK_STATE_EMPTY;

	if (RFDesc_GetActiveRFDeviceTableIndexByRFID (rfid, index))	{
		*plinkstate = g_pRFDeviceDescriptiveTable[index[0]].linkstate;
		res = SUCCESS;
	}	else {
		for (i=0; i<g_MaxSupportedRFDevices; i++)	{
			if (g_pRFDeviceDescriptiveTable[i].IDInfo.rfid == rfid)	{
				*plinkstate = g_pRFDeviceDescriptiveTable[i].linkstate;
				res = SUCCESS;
				break;
			}
		}
}

	return (res);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This routines returns list of indexes to the description table entry of given RFID
 *
 * \detail
 * \date        08/24/2006
 *
 * \param	    UINT8 rfid, UINT8 *pindextable
 *
 * \return		UINT16 count
 *
 * \retval		Returns number of
 */
UINT16	RFDesc_GetIndexToDescriptiveTableByRFID (UINT8 rfid, UINT8 *pindextable)	{
	UINT16 i, count = 0;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable;

	for (i=0; i<g_MaxSupportedRFDevices; i++, pdst++)	{
		if ((pdst->linkstate != RF_DEVICE_LINK_STATE_EMPTY) && (pdst->linkstate != RF_DEVICE_LINK_STATE_TEMP_CONN_ACCESS))	{
			if (pdst->IDInfo.rfid == rfid)	{
				*(pindextable+count) = (UINT8)i;
				count++;
			}
		}
	}
	return (count);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This routines returns the device index to the descriptive table by TSID
 *
 * \detail
 * \date        08/24/2006
 *
 * \param	    UINT23 tsid, UINT8 *pindextable
 *
 * \return		UINT16 count
 *
 * \retval		Returns total nuber of devices found with this TSID
 */
UINT16	RFDesc_GetIndexToDescriptiveTableByTSID (UINT32 tsid, UINT8 *pindextable)	{
	UINT16 i, count = 0;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable;

	for (i=0; i<g_MaxSupportedRFDevices; i++, pdst++)	{
		if ((pdst->linkstate != RF_DEVICE_LINK_STATE_EMPTY) && (pdst->linkstate != RF_DEVICE_LINK_STATE_TEMP_CONN_ACCESS))	{
			if (pdst->IDInfo.tsid == tsid)	{
				*(pindextable+count) = (UINT8)i;
				count++;
			}
		}
	}
	return (count);
}

/**
 *
 * \author      Hazrat Shah
 *
 * \brief		This routines returns the device index to the descriptive table by TSID
 *
 * \detail
 * \date        08/24/2006
 *
 * \param	    UINT23 tsid, UINT8 *pindextable, UINT8 linkstate
 *
 * \return		UINT16 count
 *
 * \retval		Returns total nuber of devices found with this TSID
 */
UINT16	RFDesc_GetIndexToDescriptiveTableByTSIDWithGivenLinkState (UINT32 tsid, UINT8 *pindextable, UINT8 linkstate)	{
	UINT16 i, count = 0;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable;

	for (i=0; i<g_MaxSupportedRFDevices; i++, pdst++)	{
		if (pdst->linkstate == linkstate)	{
			if (pdst->IDInfo.tsid == tsid)	{
				*(pindextable+count) = (UINT8)i;
				count++;
			}
		}
	}
	return (count);	
}


/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routines returns the device index to the descriptive table by UID    	 			           
 *
 * \detail		
 * \date        08/24/2006
 *
 * \param	    UINT8 *puid, UINT8 *pindex
 *
 * \return		UINT16 count 			
 *
 * \retval		Returns 0 if no device was found with this UID     
 */
UINT16	RFDesc_GetIndexToDescriptiveTableByUID (UINT8 *puid, UINT8 *pindex)	{
	UINT16 i, count = 0;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable;
	
	for (i=0; i<g_MaxSupportedRFDevices; i++, pdst++)	{
		if ((pdst->linkstate != RF_DEVICE_LINK_STATE_EMPTY) && (pdst->linkstate != RF_DEVICE_LINK_STATE_TEMP_CONN_ACCESS))	{
			if (memcmp((UINT8*)&pdst->IDInfo.uid, puid, UID_MAX_SIZE) == 0)	{
				/* Found the UID, break out. */
				*(pindex) = (UINT8)i;
				count = 1;
				break;
			}
		}				
	}		
	return (count);	
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routines returns the device index to the descriptive table by UID    	 			           
 *
 * \detail		
 * \date        08/24/2006
 *
 * \param	    UINT8 *puid, UINT8 *pindex, UINT8 linkstate 
 *
 * \return		UINT16 count 			
 *
 * \retval		Returns 0 if no device was found with this UID     
 */
UINT16	RFDesc_GetIndexToDescriptiveTableByUIDWithGivenLinkState (UINT8 *puid, UINT8 *pindex, UINT8 linkstate)	{
	UINT16 i, count = 0;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable;
	
	for (i=0; i<g_MaxSupportedRFDevices; i++, pdst++)	{
		if (pdst->linkstate == linkstate)	{
			if (memcmp((UINT8*)&pdst->IDInfo.uid, puid, UID_MAX_SIZE) == 0)	{
				/* Found the UID, break out. */
				*(pindex) = (UINT8)i;
				count = 1;
				break;
			}
		}				
	}		
	return (count);	
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routines returns the device index to the descriptive table for 
 *				all devices (exception: EMPTY and TEMP_CONN)  	 			           
 *
 * \detail		
 * \date        08/24/2006
 *
 * \param	    UINT8 *pindextable, UINT8 *puid
 *
 * \return		UINT16 count 			
 *
 * \retval		Returns total number of devices found with this UID    
 */
UINT16	RFDesc_GetIndexToDescriptiveTableAllDevices (UINT8 *pindextable)	{
	UINT16 i, count = 0;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable;
	
	for (i=0; i<g_MaxSupportedRFDevices; i++, pdst++)	{
		if ((pdst->linkstate != RF_DEVICE_LINK_STATE_EMPTY) && (pdst->linkstate != RF_DEVICE_LINK_STATE_TEMP_CONN_ACCESS))	{
			*(pindextable+count) = (UINT8)i;
			count++;			
		}				
	}		
	return (count);	
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routines returns the device index to the descriptive table for 
 *				all devices      	 			           
 *
 * \detail		
 * \date        08/24/2006
 *
 * \param	    UINT8 *pindextable, UINT8 *plinkstate, UINT8 linkstatelen
 *
 * \return		UINT16 count 			
 *
 * \retval		Returns total number of devices found with this UID    
 */
UINT16	RFDesc_GetIndexToDescriptiveTableDevicesExceptForGivenStates (UINT8 *pindextable, UINT8 *plinkstate, UINT8 linkstatelen)	{
	UINT16 i, count = 0;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable;
	UINT32	lklist, ignore;

	for (i=0; i<g_MaxSupportedRFDevices; i++, pdst++)	{
		lklist = 0;
		ignore = FALSE;
		while (lklist < linkstatelen)	{ 
			if (*(plinkstate+lklist) == pdst->linkstate)	{
				ignore = TRUE;
				break;	
			}
			lklist++;
		}

		if (!ignore)	{
			*(pindextable+count) = (UINT8)i;
			count++;			
		}				
	}		
	return (count);	
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routines returns the device index to the descriptive table for 
 *				all devices which given link state	    	 			           
 *
 * \detail		
 * \date        05/18/2007
 *
 * \param	    UINT8 *pindextable, UINT8 linkstate 
 *
 * \return		UINT16 count 			
 *
 * \retval		Returns total number of devices found with this UID    
 */
UINT16	RFDesc_GetIndexToDescriptiveTableAllDevicesWithGivenLinkState (UINT8 *pindextable, UINT8 linkstate)	{
	UINT16 i, count = 0;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable;
	
	for (i=0; i<g_MaxSupportedRFDevices; i++, pdst++)	{
		if (pdst->linkstate == linkstate)	{
			*(pindextable+count) = (UINT8)i;
			count++;			
		}				
	}		

	return (count);	
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routines returns index of the description table entry of given UID 	 			           
 *
 * \detail		
 * \date        08/24/2006
 *
 * \param	    UINT8 index, UINT32 tsid	
 *
 * \return		UINT16 duplicate 			
 *
 * \retval		Returns SUCCESS if duplicate UID found	
 */
UINT16	RFDesc_IsDuplicateUID(UINT8 index, UINT32 tsid)	{
	UINT16 i, duplicate = FAILURE;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst    = g_pRFDeviceDescriptiveTable;
	TYPE_RF_DESCRIPTIVE_TABLE *pdevdst = (g_pRFDeviceDescriptiveTable+index);
	
	for (i=0; i<g_MaxSupportedRFDevices; i++, pdst++)	{
		if ((pdst->linkstate != RF_DEVICE_LINK_STATE_TEMP_CONN_ACCESS) && (pdst->linkstate != RF_DEVICE_LINK_STATE_EMPTY))	{
			if ((memcmp ((INT8 *) &pdst->IDInfo.uid, (INT8 *) &pdevdst->IDInfo.uid, UID_MAX_SIZE) == 0) && (pdevdst->IDInfo.tsid != tsid))	{
				//	duplicate uid				
				duplicate = SUCCESS;	
				break;
			}
		}				
	}			
	
	return (duplicate);	
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routines copies the rf device id string to the device descriptor table	  
 *
 * \detail				
 *
 * \date        10/11/2006
 *
 * \param	    UINT8 index, UINT8 *pidstr, UINT8 idstrlen		
 *
 * \return		void 			
 *
 * \retval		None
 */
void	RFDesc_UpdateDeviceIDStr (UINT8 index, UINT8 *pidstr, UINT8 idstrlen)	{
	TYPE_RF_DESCRIPTIVE_TABLE *pdst    = (g_pRFDeviceDescriptiveTable+index);
	
	//	check for valid string len 
	if (idstrlen >= (sizeof(pdst->IDInfo.idstring)))	{	
		idstrlen = sizeof(pdst->IDInfo.idstring)-1;			
	}	
	memset (pdst->IDInfo.idstring, 0, sizeof(pdst->IDInfo.idstring));
	memcpy (pdst->IDInfo.idstring, pidstr, idstrlen);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routines copies the rf device id paramters to the descriptor table  
 *
 * \detail				
 *
 * \date        10/11/2006
 *
 * \param	    UINT8 index, TYPE_RF_LINK_REQUEST_PACKET_DATA *pdata, UINT8 idstrlen		
 *
 * \return		void 			
 *
 * \retval		None	
 */
void	RFDesc_UpdateDeviceIDInfo(UINT8 index, UINT8 power, UINT32 tsid, UINT8 rfid, UINT16 panid, UINT8 options, UINT8 *pidstr, UINT8 idstrlen)	{
	TYPE_RF_DESCRIPTIVE_TABLE *pdst    = (g_pRFDeviceDescriptiveTable+index);

	pdst->IDInfo.power = power;
	pdst->IDInfo.tsid  = tsid;
	pdst->IDInfo.rfid  = rfid;
	pdst->IDInfo.panid = panid;
	pdst->IDInfo.deviceoptions = options;
	RFDesc_UpdateDeviceIDStr (index, pidstr, idstrlen);	
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routines searches for given descriptive table entry type 
 *
 * \detail				
 *
 * \date        10/11/2006
 *
 * \param	    UINT8 *pdsindex, UINT8 linkstate		
 *
 * \return		UINT16 found 			
 *
 * \retval		SUCCESS/FAILURE	
 */
UINT16 RFDesc_SearchForDescriptiveEntrywithGivenLinkState (UINT8 *pdsindex, UINT8 linkstate)	{
	UINT16	i, found = FAILURE;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable;
	
	for (i=0; i<g_MaxSupportedRFDevices; i++, pdst+=1)	{
		if (pdst->linkstate == linkstate)	{
			*pdsindex = (UINT8)i;				
			found     = SUCCESS;
			break;
		}				
	}			
	
	return (found);	
}
#ifndef	RF_NETWORK_MASTER_DEVICE	
/**
 *
 * \author      john Cheng	
 *
 * \brief		This routines init. an entry in the Roaming descriptive 
 *              table
 *
 * \detail		
 *
 * \date        12/11/2008
 *
 * \param	    UINT8 index, UINT8 *puid, UINT16 panid, UINT8 addrmode, UINT8 rxsequence, UINT8 linkstate 			
 *
 * \return		always SUCCESS for now			
 *
 * \retval		SUCCESS	
 */
UINT16 RFDesc_InitEntryInRoamingDescriptiveTable (UINT8 index, UINT8 *puid, UINT16 panid, UINT8 addrmode, UINT8 rxsequence, UINT8 linkstate)	
{
	UINT16 res = SUCCESS;

    g_pRFRoamingDescriptiveTable[index].linkstate = linkstate;
	memcpy ((void*) g_pRFRoamingDescriptiveTable[index].IDInfo.uid, (void*) puid, UID_MAX_SIZE); 		
	g_pRFRoamingDescriptiveTable[index].IDInfo.panid    = panid;
	g_pRFRoamingDescriptiveTable[index].IDInfo.addrmode = addrmode;
	//	Syncronize the sequence count 
	g_pRFRoamingDescriptiveTable[index].rxsequence      = 0;//use this rxsequence to tell how many received	
	g_pRFRoamingDescriptiveTable[index].RxBuf.index     = 0;//use this index to save LQ in buffer
	memset ((void*)g_pRFRoamingDescriptiveTable[index].RxBuf.data, 0, sizeof(g_pRFRoamingDescriptiveTable[index].RxBuf.data));
	return (res);
}
#endif
/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routines creates an entry in the descriptive table	
 *
 *					Creation order:	
 *						1=Devices with link state Empty 
 *						2=Devices with link state Power Down 
 *						3=Devices with link state Sleep 	 			           
 *						4=Devices with link state Fail
 *
 * \detail		
 *
 * \date        10/11/2006
 *
 * \param	    UINT8 *pdsindex, UINT8 *puid, UINT16 panid, UINT8 rxsequence 			
 *
 * \return		UINT16 found  			
 *
 * \retval		SUCCESS/FAILURE	
 */
UINT16 RFDesc_InitEntryInDescriptiveTable (UINT8 index, UINT8 *puid, UINT16 panid, UINT8 addrmode, UINT8 rxsequence, UINT8 linkstate)	{
	UINT16 res = FAILURE;

	if (index < g_MaxDescriptiveTableEntries)	{
		RFDesc_LinkStateTransition (index, linkstate);	
		memcpy ((void*) g_pRFDeviceDescriptiveTable[index].IDInfo.uid, (void*) puid, UID_MAX_SIZE); 		
		g_pRFDeviceDescriptiveTable[index].IDInfo.panid    = panid;
		g_pRFDeviceDescriptiveTable[index].IDInfo.addrmode = addrmode;
		//	Syncronize the sequence count 
		g_pRFDeviceDescriptiveTable[index].rxsequence      = rxsequence;		
	}

	return (res);
}	   
#ifndef	RF_NETWORK_MASTER_DEVICE
/**
 *
 * \author      John Cheng	
 *
 * \brief		This routines Release an entry in the Roaming descriptive 
 *              table
 *
 * \detail		if index is not with 1 to 8 range, it will reset all entries
 *
 * \date        12/11/2008
 *
 * \param	    UINT8 index 			
 *
 * \return		void			
 *
 * \retval		none	
 */
void RFDesc_ReleaseEntryInRoamingDescriptiveTable (UINT8 index)
{
    //search through roaming Descriptive Table first
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFRoamingDescriptiveTable;
	if(index >= MAX_ROAMING_DESCRIPTIVE_TABLE_ENTRIES || index < 0)
	{
		int i;
		for (i=0; i<MAX_ROAMING_DESCRIPTIVE_TABLE_ENTRIES; i++, pdst+=1)
		{
            memset ((void*) pdst, 0, sizeof(TYPE_RF_DESCRIPTIVE_TABLE));
		}	
		return;
	}
	else
	{   
		memset ((void*) &pdst[index], 0, sizeof(TYPE_RF_DESCRIPTIVE_TABLE));
	}
}
/**
 *
 * \author      John Cheng	
 *
 * \brief		This routines check to see if there is any acquired gateway 
 *              in the roaming table
 *
 * \detail		it returns true if there is any
 *
 * \date        12/11/2008
 *
 * \param	    UINT8 index 			
 *
 * \return		BOOL			
 *
 * \retval		TRUE/FALSE	
 */
BOOL RFDesc_RoamingDescriptiveTableIsEmpty()
{
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFRoamingDescriptiveTable;
	int i;
	for ( i = 0; i < MAX_ROAMING_DESCRIPTIVE_TABLE_ENTRIES; i++, pdst += 1 )
    {
		if (pdst->linkstate != RF_DEVICE_LINK_STATE_EMPTY )
			return FALSE;
	}
	return TRUE;
}
/**
 *
 * \author      John Cheng	
 *
 * \brief		This routines creates an entry in the roaming table
 *
 * \detail		it returns SUCCESS if it created,else FAILURE
 *
 * \date        12/11/2008
 *
 * \param	    UINT8 *pdsindex, UINT8 *puid, UINT16 panid, UINT8 addrmode, UINT8 rxsequence 			
 *
 * \return		UINT16			
 *
 * \retval		SUCCESS/FAILURE	
 */
UINT16 RFDesc_CreateEntryInRoamingDescriptiveTable (UINT8 *pdsindex, UINT8 *puid, UINT16 panid, UINT8 addrmode, UINT8 rxsequence)
{
	UINT16 found = FAILURE;
	int GW_slot = g_acquireSlot;
	int i;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFRoamingDescriptiveTable;

	if( GW_slot < 1 || GW_slot > MAX_ROAMING_DESCRIPTIVE_TABLE_ENTRIES )
	{
		GW_slot = 1;
	}
	GW_slot--;

    for ( i = 0; i < MAX_ROAMING_DESCRIPTIVE_TABLE_ENTRIES; i++, pdst += 1 )
    {
		if( i == GW_slot ) /* this entry is empty already when acquire started */
			continue;

		if (pdst->linkstate != RF_DEVICE_LINK_STATE_EMPTY )
		{
			if( memcmp((UINT8*)&pdst->IDInfo.uid, puid, UID_MAX_SIZE) == 0 )//same UID
			{
				UINT8* uidPtr = (UINT8*)&pdst->IDInfo.uid;
                return (FAILURE);
			}
		}
	}

	*pdsindex = GW_slot;
	if( g_pRFRoamingDescriptiveTable[GW_slot].linkstate == RF_DEVICE_LINK_STATE_EMPTY )
	{
		RFDesc_InitEntryInRoamingDescriptiveTable (GW_slot, puid, panid, addrmode, rxsequence, RF_DEVICE_LINK_STATE_TEMP_CONN_ACCESS);
	}
    found = SUCCESS;

	return (found);
}
/**
 *
 * \author      John Cheng	
 *
 * \brief		This routines returns the index to the roaing table with given 
 *              gateway UID 
 *
 * \detail		it returns true if there is any
 *
 * \date        12/11/2008
 *
 * \param	    UINT8 *puid, UINT8 *pindextable			
 *
 * \return		BOOL			
 *
 * \retval		TRUE/FALSE	
 */
BOOL	RFDesc_GetIndexToRoamingDescriptiveTableByUID (UINT8 *puid, UINT8 *pindextable)	
{
	//UINT8 *uid;
	int i;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFRoamingDescriptiveTable;
	for (i=0; i<MAX_ROAMING_DESCRIPTIVE_TABLE_ENTRIES; i++, pdst++)	
	{
		if ((pdst->linkstate != RF_DEVICE_LINK_STATE_EMPTY) )	
		{
			if (memcmp((UINT8*)&pdst->IDInfo.uid, puid, UID_MAX_SIZE) == 0)	{
                *pindextable = i;
				return TRUE;
			}

		}				
	}		
	return FALSE;	
}
#endif
/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routines creates an entry in the descriptive table	
 *
 *					Creation order:	
 *						1=Devices with link state Empty 
 *						2=Devices with link state Power Down 
 *						3=Devices with link state Sleep 	 			           
 *						4=Devices with link state Fail
 *
 * \detail		
 *
 * \date        10/11/2006
 *
 * \param	    UINT8 *pdsindex, UINT8 *puid, UINT16 panid, UINT8 rxsequence 			
 *
 * \return		UINT16 found  			
 *
 * \retval		SUCCESS/FAILURE	
 */
UINT16 RFDesc_CreateEntryInDescriptiveTable (UINT8 *pdsindex, UINT8 *puid, UINT16 panid, UINT8 addrmode, UINT8 rxsequence)	{
	UINT16 found = FAILURE;

	if 			(SUCCESS == RFDesc_SearchForDescriptiveEntrywithGivenLinkState (pdsindex, RF_DEVICE_LINK_STATE_EMPTY)){				
		found = SUCCESS;
	}	else if (SUCCESS == RFDesc_SearchForDescriptiveEntrywithGivenLinkState (pdsindex, RF_DEVICE_LINK_STATE_POWERDOWN)){	
		found = SUCCESS;	
	}	else if (SUCCESS == RFDesc_SearchForDescriptiveEntrywithGivenLinkState (pdsindex, RF_DEVICE_LINK_STATE_SLEEP)){	
		found = SUCCESS;	
	}	else if (SUCCESS == RFDesc_SearchForDescriptiveEntrywithGivenLinkState (pdsindex, RF_DEVICE_LINK_STATE_FAIL))	{	
		found = SUCCESS;
	}	

	if (found == SUCCESS)	{
		RFDesc_InitEntryInDescriptiveTable (*pdsindex, puid, panid, addrmode, rxsequence, RF_DEVICE_LINK_STATE_TEMP_CONN_ACCESS);	
	}

	return (found);
}	   

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routines creates an entry in the descriptive table	
 *
 * \detail		
 *
 * \date        10/11/2006
 *
 * \param	    UINT8 *pdsindex, UINT8 *puid, UINT16 panid, UINT8 addrmode			
 *
 * \return		UINT16 found  			
 *
 * \retval		SUCCESS/FAILURE	
 */
UINT16 RFDesc_GetNetworkServiceDescriptiveTableEntryByUID (UINT8 *pdsindex, UINT8 *puid, UINT16 panid, UINT8 addrmode)	{
	UINT16 found = FAILURE;
	
	*pdsindex = RFDesc_GetNetworkServicesDescriptiveTableIndex();

	if (g_pRFDeviceDescriptiveTable[*pdsindex].linkstate != RF_DEVICE_LINK_STATE_EMPTY)	{
		if (!memcmp(g_pRFDeviceDescriptiveTable[*pdsindex].IDInfo.uid, puid, UID_MAX_SIZE) && 
			(g_pRFDeviceDescriptiveTable[*pdsindex].IDInfo.panid == panid) &&
			(g_pRFDeviceDescriptiveTable[*pdsindex].IDInfo.addrmode == addrmode))	{	
			found = SUCCESS;
		}
	}

	return (found);
}	   

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This routines creates an entry in the descriptive table	
 *
 * \detail		
 *
 * \date        10/11/2006
 *
 * \param	    UINT8 *pdsindex, UINT8 *puid, UINT16 panid, UINT8 rxsequence			
 *
 * \return		UINT16 found  			
 *
 * \retval		SUCCESS/FAILURE	
 */
UINT16 RFDesc_CreateNetworkServiceDescriptiveTableEntryByUID (UINT8 *pdsindex, UINT8 *puid, UINT16 panid, UINT8 addrmode, UINT8 rxsequence)	{
	UINT16 found = FAILURE;
	
	*pdsindex = RFDesc_GetNetworkServicesDescriptiveTableIndex();

	//if (g_pRFDeviceDescriptiveTable[*pdsindex].linkstate == RF_DEVICE_LINK_STATE_EMPTY)	{
		RFDesc_InitEntryInDescriptiveTable (*pdsindex, puid, panid, addrmode, rxsequence, RF_DEVICE_LINK_STATE_TEMP_CONN_ACCESS);	
		found = SUCCESS;
	//}

	return (found);
}	   

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function builds a list of indexes to the entry table  					     	 			           
 *
 * \detail		The RF ID can be used as an index to lookup the table entry in the descriptive table 
 *				There could be multiple entries for the same RFID in the descriptive table. This will 
 *				usually occur at setup time.				  															
 *				This table should be updated when a new device is found or a device is lost	 
 *
 * \date        08/24/2006
 * \param	    none		 	
 *
 * \return		none				
 *
 * \retval		                
 */				
void RFDesc_BuildActiveRFDeviceTableIndexByRFID (void)	{		
	UINT16	i;	 					   
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable;
#if defined (OS_WINCE)
	LONG lPreviousCount;	// for the sake of function, but not used.
#endif

#if defined (OS_NUCLEUS)
	if (NU_SUCCESS == NU_Obtain_Semaphore(&RFDevByRFID_Semaphore, RFDEVBYRFID_SEMAPHORE_TIMEOUT)) {
#elif defined (OS_WINCE)
	if (WAIT_OBJECT_0 == WaitForSingleObject(RFDevByRFID_Semaphore, RFDEVBYRFID_SEMAPHORE_TIMEOUT)) {
#endif
	if(OSTakeCountingSemaphore(RFDevByRFID_Semaphore, RFDEVBYRFID_SEMAPHORE_TIMEOUT_MS)) {
		//	Clear active device list
		memset ((UINT8 *) &RFDevByRFID, 0, sizeof(RFDevByRFID));
		
		for (i=0; i<g_MaxSupportedRFDevices; i++, pdst++)	{
			if ((pdst->linkstate == RF_DEVICE_LINK_STATE_ACTIVE) && (pdst->IDInfo.rfid < sizeof(RFDevByRFID[0].index)))	{ 
				RFDevByRFID[pdst->IDInfo.rfid].index[RFDevByRFID[pdst->IDInfo.rfid].count] = (UINT8)i;
				RFDevByRFID[pdst->IDInfo.rfid].count++;	 
			}				
		}
#if defined (OS_NUCLEUS)
		NU_Release_Semaphore(&RFDevByRFID_Semaphore);	
#elif defined (OS_WINCE)
		ReleaseSemaphore(RFDevByRFID_Semaphore, 1, &lPreviousCount);
#endif
		OSGiveCountingSemaphore(RFDevByRFID_Semaphore);
	}	else	{
		RFIntHostApp.pF_ErrorHandler("RFDesc:Could not obtain the RFDeviceTable index Semaphore");
	}
}
	
/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns index to the descriptive table for given RFID   					     	 			           
 *
 * \detail		
 *
 * \date        08/24/2006
 * \param	    RFID, pointer to index entries			 	
 *
 * \return		indexcnt				
 *
 * \retval		                
 */				
UINT16 RFDesc_GetActiveRFDeviceTableIndexByRFID (UINT8 rfid, UINT8 *pIndex)	{		
	UINT16	count = 0;

#ifdef	RF_NETWORK_MASTER_DEVICE
	UINT16 i;
 #if defined (OS_WINCE)
	LONG lPreviousCount;
 #endif
#endif
				
#ifdef	RF_NETWORK_MASTER_DEVICE
 #if defined (OS_NUCLEUS)
	if (NU_SUCCESS == NU_Obtain_Semaphore(&RFDevByRFID_Semaphore, RFDEVBYRFID_SEMAPHORE_TIMEOUT))	{
 #elif defined (OS_WINCE)
	if (WAIT_OBJECT_0 == WaitForSingleObject(RFDevByRFID_Semaphore, RFDEVBYRFID_SEMAPHORE_TIMEOUT)) {
 #endif 
		//	If rfid is broadcast id then add all rf devices to the list
		if (rfid == RFS_CNET_BROADCAST)	{
			for (i=0, count=0; i<sizeof(RFDevByRFID[0].index); i++)	{  
				if (RFDevByRFID[i].count)	{
					memcpy (pIndex, (UINT8*)RFDevByRFID[i].index, RFDevByRFID[i].count);
					count += RFDevByRFID[i].count;										
					pIndex = pIndex + RFDevByRFID[i].count;
				}
			}		
		}	else	{
			//	only add the specific rf devices to the list
			if (RFDevByRFID[rfid].count)	{
				memcpy (pIndex, (UINT8*) &RFDevByRFID[rfid].index, RFDevByRFID[rfid].count);
			    count   = RFDevByRFID[rfid].count;
			}
		}
  #if defined (OS_NUCLEUS)
		NU_Release_Semaphore(&RFDevByRFID_Semaphore);
  #elif defined (OS_WINCE)
		ReleaseSemaphore(RFDevByRFID_Semaphore, 1, &lPreviousCount);
  #endif
	}	else	{
		RFIntHostApp.pF_ErrorHandler("RFDesc:Could not obtain the RFDeviceTable index Semaphore");
	}
#else
	*pIndex = DEVICE_DESCRIPTIVE_TABLE_INDEX;
	//	HS - the host application determines on whether to send packets if link is not up 
	//if (RFReceive_IsGatewayConnected())	{	count = 1;	}	else	{	count = 0;	}
	count = 1;
#endif

	return (count);
}



////////////////////////////////////////////////////////////////////////////////
//	GLOBAL FUNCTIONS 
//////////////////////////////////////////////////////////////////////////////

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns the RF MLME packet index in the descriptive table      			           
 *			
 *
 * \detail		
 *						
 * \date        09/15/2006
 *
 * \param	    None
 *
 * \return		None			
 *
 * \retval		
 */
UINT8 	RFDesc_GetMLMEDescriptiveTableIndex (void)	{
	return (g_MaxDescriptiveTableEntries-1);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns the RF ASP packet index in the descriptive table      			           
 *			
 *
 * \detail		
 *						
 * \date        09/15/2006
 *
 * \param	    None
 *
 * \return		None			
 *
 * \retval		
 */
UINT8 	RFDesc_GetASPDescriptiveTableIndex (void)	{
	return (g_MaxDescriptiveTableEntries-2);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function returns the RF Network Services descriptive table index       			           
 *			
 *
 * \detail		
 *						
 * \date        08/02/2007
 *
 * \param	    None
 *
 * \return		None			
 *
 * \retval		
 */
UINT8 	RFDesc_GetNetworkServicesDescriptiveTableIndex (void)	{
	return (g_MaxDescriptiveTableEntries-3);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function initializes the max count of supported rf devices     			           
 *
 * \detail		
 *						
 * \date        09/11/2006
 * \param	    None
 *
 * \return		None			
 *
 * \retval		
 */
void RFDesc_LocalDeviceParamInit (void)	{	
#ifdef	RF_NETWORK_MASTER_DEVICE
	EEpromGetMaxSupportedRFDeviceCount (&g_MaxSupportedRFDevices);
#else	
	//	The RF Slave device only communicates with the gateway 
	g_MaxSupportedRFDevices = 1;
#endif

	if (g_MaxSupportedRFDevices > MAX_RF_NETWORK_SUPPORTED_DEVICES)	{
		g_MaxSupportedRFDevices = MAX_RF_NETWORK_SUPPORTED_DEVICES;
	}	

	//	Minimum one device should be supported 
	if (g_MaxSupportedRFDevices == 0)	{	g_MaxSupportedRFDevices = 1;	}
}
							 							 
/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function is called on startup to initialize various rf parameters           
 *
 * \detail		This allocates descriptor table, creates pipe and semaphore for devices 
 * 
 * \date        08/22/2006
 * \param	    None
 *
 * \return		None			
 *
 * \retval		
 */
void RFDesc_Init (void)	{
    UINT32	i, maxentries;
	const TYPE_INTERFACE_TO_HOST_APP *pHF = &RFIntHostApp;
//	BOOL	valid;
	UINT32	pipesize;

	HPRF_DEBUG(DmConsolePrintf("%s\n", __FUNCTION__);)

#if defined (OS_WINCE)
	MSGQUEUEOPTIONS sMsgQueueOpt;
	TCHAR msgString[32];
#endif

	//	initialize rf net parameters shadow ram 
	RFDesc_LocalDeviceParamInit();

	//	Define Descriptor entries 
	//		x table entries for devices as for cfg file
	//		1 table entry for diagnostic packets   
	//		1 table entry for ASP packet, 
	//		1 table entry for MLME packet		
	maxentries =  g_MaxSupportedRFDevices+3;

#ifdef	RF_NETWORK_MASTER_DEVICE
	//	todo:	set pipesize based on device types (high/med/low priority) and number of each device type
	pipesize   =  (GATEWAY_MEMORY_SIZE_FOR_ALL_PIPES / maxentries);
	if (pipesize >= GATEWAY_MAX_ALLOWABLE_MEMORY_PER_PIPE)	{	pipesize = GATEWAY_MAX_ALLOWABLE_MEMORY_PER_PIPE;	}	
#else	
	pipesize   =  (SLAVE_DEVICE_MEMORY_SIZE_FOR_ALL_PIPES / maxentries);
	if (pipesize >= SLAVE_DEVICE_MAX_ALLOWABLE_MEMORY_PER_PIPE)	{	pipesize = SLAVE_DEVICE_MAX_ALLOWABLE_MEMORY_PER_PIPE;	}	
#endif

#if defined (OS_NUCLEUS)
	//	Allocate memory for the rf device descriptor table
	if (NU_SUCCESS != NU_Allocate_Memory(&System_Memory, (void*) &g_pRFDeviceDescriptiveTable, sizeof(TYPE_RF_DESCRIPTIVE_TABLE)*maxentries, NU_NO_SUSPEND))	{
		RFIntHostApp.pF_SystemError (FATAL_RF_INIT); 							
	}
#ifndef	RF_NETWORK_MASTER_DEVICE
	//	Allocate memory for the rf device descriptor table 
	if (NU_SUCCESS != NU_Allocate_Memory(&System_Memory, (void*) &g_pRFRoamingDescriptiveTable, sizeof(TYPE_RF_DESCRIPTIVE_TABLE)*MAX_ROAMING_DESCRIPTIVE_TABLE_ENTRIES, NU_NO_SUSPEND))	{
		RFIntHostApp.pF_SystemError (FATAL_RF_INIT); 							
	}
	memset ((void*) g_pRFRoamingDescriptiveTable, 0, sizeof(TYPE_RF_DESCRIPTIVE_TABLE)*MAX_ROAMING_DESCRIPTIVE_TABLE_ENTRIES);
#endif
	//	Clear descriptor table memory
	memset ((void*) g_pRFDeviceDescriptiveTable, 0, sizeof(TYPE_RF_DESCRIPTIVE_TABLE)*maxentries);

	//	Allocate memory for the rf device descriptor table pipes 	
	for (i=0; i<maxentries; i++)
	{
		if (NU_SUCCESS != NU_Allocate_Memory(&System_Memory, (void*) &g_pRFDeviceDescriptiveTable[i].ppipebuf, pipesize, NU_NO_SUSPEND))	{
			RFIntHostApp.pF_SystemError (FATAL_RF_INIT); 							
		}
		if (NU_Create_Pipe(&g_pRFDeviceDescriptiveTable[i].pipe, "PRFDESC", g_pRFDeviceDescriptiveTable[i].ppipebuf, 
							pipesize, NU_VARIABLE_SIZE, sizeof(TYPE_RF_DEVICE_RF_DATA_HOLD_BUF), NU_FIFO) != NU_SUCCESS)	{
	   		RFIntHostApp.pF_SystemError (FATAL_RF_INIT);    		   
		}
	}
	if (NU_SUCCESS != NU_Create_Semaphore(&RFDevByRFID_Semaphore, "SRFMANAG", 1, NU_FIFO))	{
		RFIntHostApp.pF_SystemError (FATAL_RF_INIT);
    }
	NU_Reset_Semaphore(&RFDevByRFID_Semaphore, 1);
	//	Descriptor entries were created 
	g_MaxDescriptiveTableEntries =  maxentries;
#elif defined (OS_WINCE)
	//(HANDLE) g_pRFDeviceDescriptiveTable = HeapAlloc(hprSystemMemoryPool,
	//												 LPTR, (sizeof(TYPE_RF_DESCRIPTIVE_TABLE)* maxentries));
	(HANDLE) g_pRFDeviceDescriptiveTable = LocalAlloc(LPTR, (sizeof(TYPE_RF_DESCRIPTIVE_TABLE)* maxentries));
	if (g_pRFDeviceDescriptiveTable == NULL)
	{
		RFIntHostApp.pF_ErrorPrintf("RFDesc_Init: Fail desc. table alloc");
		RFIntHostApp.pF_SystemError (FATAL_RF_INIT);
	}
 #ifndef	RF_NETWORK_MASTER_DEVICE	
	//(HANDLE) g_pRFRoamingDescriptiveTable = HeapAlloc(hprSystemMemoryPool,
	//												 LPTR, (sizeof(TYPE_RF_DESCRIPTIVE_TABLE)* MAX_ROAMING_DESCRIPTIVE_TABLE_ENTRIES));
	(HANDLE) g_pRFRoamingDescriptiveTable = LocalAlloc(LPTR, (sizeof(TYPE_RF_DESCRIPTIVE_TABLE)* MAX_ROAMING_DESCRIPTIVE_TABLE_ENTRIES));
	if (g_pRFRoamingDescriptiveTable == NULL)
	{
		RFIntHostApp.pF_ErrorPrintf("RFDesc_Init: Fail roaming des. table alloc");
		RFIntHostApp.pF_SystemError (FATAL_RF_INIT);
	}
	//memset ((void*) g_pRFRoamingDescriptiveTable, 0, sizeof(TYPE_RF_DESCRIPTIVE_TABLE)*MAX_ROAMING_DESCRIPTIVE_TABLE_ENTRIES);
 #endif
	//	Clear descriptor table memory
	//memset ((void*) g_pRFDeviceDescriptiveTable, 0, sizeof(TYPE_RF_DESCRIPTIVE_TABLE)*maxentries);


	//Set options for the message queue
	sMsgQueueOpt.dwSize = sizeof(sMsgQueueOpt);		//Size of the structure in bytes
	sMsgQueueOpt.dwFlags = 0;						//Describes the behavior of the message queue
	sMsgQueueOpt.dwMaxMessages = 0;					//No limit on number of messages to queue at any one time, set to zero
	sMsgQueueOpt.cbMaxMessage = sizeof(TYPE_RF_DEVICE_RF_DATA_HOLD_BUF); //Maximum number of bytes in each message
	
	//	Allocate memory for the rf device descriptor table pipes
	for (i=0; i<maxentries; i++)
	{
		//(HANDLE) g_pRFDeviceDescriptiveTable[i].ppipebuf = HeapAlloc(hprSystemMemoryPool, LPTR, pipesize);
		//if (g_pRFDeviceDescriptiveTable[i].ppipebuf == NULL)
		//{
		//	RFIntHostApp.pF_SystemError (FATAL_RF_INIT);
		//}
		_stprintf(msgString, _T("HPRF%d"), i);

		// Write handle to queue.
		sMsgQueueOpt.bReadAccess = FALSE;
		g_pRFDeviceDescriptiveTable[i].QueueWriteHandle = CreateMsgQueue(msgString,  &sMsgQueueOpt);
		if (g_pRFDeviceDescriptiveTable[i].QueueWriteHandle == NULL)
		{
			RFIntHostApp.pF_ErrorPrintf("RFDesc_Init: Fail create WriteQ: %d", i);
		}

		// Read handle to queue
		sMsgQueueOpt.bReadAccess = TRUE;
		g_pRFDeviceDescriptiveTable[i].QueueReadHandle = CreateMsgQueue(msgString,  &sMsgQueueOpt);
		if (g_pRFDeviceDescriptiveTable[i].QueueReadHandle == NULL)
		{
			RFIntHostApp.pF_ErrorPrintf("RFDesc_Init: Fail create ReadQ: %d", i);
		}
		
	}
	RFDevByRFID_Semaphore = CreateSemaphore(NULL, 1, 1, TEXT("SRFMANAG"));
	if (NULL == RFDevByRFID_Semaphore)
	{
		RFIntHostApp.pF_ErrorPrintf("RFDesc_Init: Fail create Semaphore");
		RFIntHostApp.pF_SystemError (FATAL_RF_INIT);
    }
	//	Descriptor entries were created 
	g_MaxDescriptiveTableEntries =  maxentries;
#endif // OS_WINCE
	g_pRFDeviceDescriptiveTable = OsCallocMemory(maxentries, sizeof(TYPE_RF_DESCRIPTIVE_TABLE));
	if (g_pRFDeviceDescriptiveTable == NULL)
	{
	    pHF->pF_ErrorPrintf("RFDesc_Init: Fail desc. table alloc");
	    pHF->pF_SystemError (FATAL_RF_INIT);
	}
 #ifndef	RF_NETWORK_MASTER_DEVICE
	g_pRFRoamingDescriptiveTable = OsCallocMemory(MAX_ROAMING_DESCRIPTIVE_TABLE_ENTRIES, sizeof(TYPE_RF_DESCRIPTIVE_TABLE));
	if (g_pRFRoamingDescriptiveTable == NULL)
	{
	    pHF->pF_ErrorPrintf("RFDesc_Init: Fail roaming des. table alloc");
	    pHF->pF_SystemError (FATAL_RF_INIT);
	}
 #endif
	//	Allocate memory for the rf device descriptor table pipes
	for (i=0; i<maxentries; i++)
	{
		//TODO: Verify that queue is created large enough.
		g_pRFDeviceDescriptiveTable[i].QueueHandle = OsQueueCreate(
				pipesize/sizeof(TYPE_RF_DEVICE_RF_DATA_HOLD_BUF),
				sizeof(TYPE_RF_DEVICE_RF_DATA_HOLD_BUF));

		if (g_pRFDeviceDescriptiveTable[i].QueueHandle == NULL)
		{
		    pHF->pF_ErrorPrintf("RFDesc_Init: Fail create queue: %d", i);
		}
	}
	//TODO: Review Counting semaphore usage.
	RFDevByRFID_Semaphore = OSCreateCountingSemaphore(0, 1);
	if (NULL == RFDevByRFID_Semaphore)
	{
	    pHF->pF_ErrorPrintf("RFDesc_Init: Fail create Semaphore");
	    pHF->pF_SystemError (FATAL_RF_INIT);
    }
	//	Descriptor entries were created
	g_MaxDescriptiveTableEntries =  maxentries;
}																		

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function sets device link state and initializes affected parameters 	   
 *
 * \detail		  
 *
 * \date        10/11/2006
 *
 * \param	    UINT8 index, UINT8 LinkState		
 *
 * \return		None	 				
 *
 * \retval		None	 	  
 */	
void	RFDesc_LinkStateTransition (UINT8 index, UINT8 LinkState)	{	
	//UINT16	i;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable+index;
	BOOL	report = FALSE;

	if ((index < g_MaxDescriptiveTableEntries) && (index != RFDesc_GetMLMEDescriptiveTableIndex()) && (index != RFDesc_GetASPDescriptiveTableIndex()))	{
		if (pdst->linkstate != LinkState)	{
			if (pdst->linkstate == RF_DEVICE_LINK_STATE_TEMP_CONN_ACCESS)	{
				if (LinkState == RF_DEVICE_LINK_STATE_FAIL)	{	
					LinkState = RF_DEVICE_LINK_STATE_EMPTY;	
				}	
			}

			if (LinkState == RF_DEVICE_LINK_STATE_FAIL)	{
#ifdef	RF_NETWORK_MASTER_DEVICE		
				pdst->maxtxpacketretries = RFReceive_GetLocalDeviceRetry();
#else
				pdst->maxtxpacketretries = RFReceive_GetLocalDeviceRetryOnConnectionReq();
#endif
			}   

			//	Store new state
			pdst->linkstate = LinkState;

			if (LinkState != RF_DEVICE_LINK_STATE_ACTIVE)	{
		#ifdef FREESCALE_802_15_4_STACK
				pdst->LQ.linkqualityAtLocalDevice  = 0;				
				pdst->LQ.linkqualityAtRemoteDevice = 0;	
		#else
				pdst->LQ.linkqualityAtLocalDevice  = 255;	//	255 translates to low LQI (num inverted)			
				pdst->LQ.linkqualityAtRemoteDevice = 255;	//	255 translates to low LQI (num inverted)
		#endif
			}

			if (LinkState != RF_DEVICE_LINK_STATE_EMPTY)	{   
				//	reset transmit pipe, data hold buffer and inform functions whose messages are in queued in pipe
				RFTransmit_ResetTxBuffers (index);
				RFTransmit_ResetTxDataHoldBuffer(index);
				pdst->waitingforACK = 0;
				pdst->resultcodeavailable = 0; 			 	
			}

			if ((pdst->linkstate == RF_DEVICE_LINK_STATE_TEMP_CONN_ACCESS) || (pdst->linkstate == RF_DEVICE_LINK_STATE_ACTIVE)
																		   || (pdst->linkstate == RF_DEVICE_LINK_STATE_CONNECTING))	{
				pdst->waitingforACK = 0;
				pdst->resultcodeavailable = 0; 			 	
				pdst->resultcode = 0;						
				
				pdst->RetryTimer = 0;					  
				pdst->RetryTimedOut = FALSE;

				pdst->HeartBeatTimer = 0;
				pdst->HeartBeatTimedOut = FALSE;
								
				pdst->ResultCodeTimer = 0;				  
				pdst->ResultCodeTimedOut = FALSE;	
				
				pdst->txpacketretrycntr  = 0;				

#ifdef	RF_NETWORK_MASTER_DEVICE		
				pdst->maxtxpacketretries = RFReceive_GetLocalDeviceRetry();
#else
				pdst->maxtxpacketretries = RFReceive_GetLocalDeviceRetryOnConnectionReq();
#endif
				if (pdst->linkstate == RF_DEVICE_LINK_STATE_TEMP_CONN_ACCESS)	{  
					memset ((UINT8 *) &pdst->COMLog,	0, sizeof(TYPE_RF_DEVICE_COM_LOG));	
					memset ((UINT8 *) &pdst->IDInfo,	0, sizeof(TYPE_RF_DEVICE_ID_INFO));
					memset ((UINT8 *) &pdst->TxBuf,		0, sizeof(TYPE_RF_DEVICE_RFTX_BUF));
					memset ((UINT8 *) &pdst->DevField,	0, sizeof(TYPE_RF_DEVICE_INFO_REC));					
			
					pdst->linkstatecnt	 = 0;				
		
					pdst->txsequence = 0;					
					pdst->rxsequence = 0;					
				}	else if (pdst->linkstate == RF_DEVICE_LINK_STATE_ACTIVE)	{
					//	Active state
					pdst->linkstatecnt += 1;
					RFHeartBeat_ReTriggerTimer (index);
					
					pdst->maxtxpacketretries = RFReceive_GetLocalDeviceRetry();

					//	Store connection times
					pdst->linkeactiveestablishedtime = 0;
					pdst->linkeactiveestablishedmonth = 0;
					pdst->linkeactiveestablishedday = 0;
					pdst->linkeactiveestablishedyear = 0;

					RFIntHostApp.pF_GetTime((long *)&pdst->linkeactiveestablishedtime);  	
					RFIntHostApp.pF_GetDate(&pdst->linkeactiveestablishedmonth, &pdst->linkeactiveestablishedday, &pdst->linkeactiveestablishedyear);   
				}
			}

		#ifdef	RF_NETWORK_MASTER_DEVICE
			report = FALSE;
			if			(pdst->linkstate == RF_DEVICE_LINK_STATE_ACTIVE)	{
				//	post device connected message to host 
				RFIntHostApp.pF_DeviceLinkFound(pdst->IDInfo.rfid);
				report = TRUE;
			}	else if ((pdst->linkstate == RF_DEVICE_LINK_STATE_FAIL)			|| (pdst->linkstate == RF_DEVICE_LINK_STATE_SLEEP) || 
						 (pdst->linkstate == RF_DEVICE_LINK_STATE_DISCONNECTED) || (pdst->linkstate == RF_DEVICE_LINK_STATE_POWERDOWN))	{
				//	post device disconnected message to host 
				RFIntHostApp.pF_DeviceLinkLost(pdst->IDInfo.rfid);
				report = TRUE;
			}

			if (report && (index != RFDesc_GetNetworkServicesDescriptiveTableIndex()))	{
				RFDesc_BuildActiveRFDeviceTableIndexByRFID();
				SetCresnetAnalogSubscriptionData (ANALOG_JOIN_FIRST_RFDEV_LINK_QUALITY_AT_HPRFGW+g_pRFDeviceDescriptiveTable[index].IDInfo.rfid, RFReceive_ConvertLinkQualityToIEEEWord(g_pRFDeviceDescriptiveTable[index].LQ.linkqualityAtLocalDevice,  LQI_DISP_OFFSET),  FALSE);
				SetCresnetAnalogSubscriptionData (ANALOG_JOIN_FIRST_RFDEV_LINK_QUALITY_AT_DEV+g_pRFDeviceDescriptiveTable[index].IDInfo.rfid,    RFReceive_ConvertLinkQualityToIEEEWord(g_pRFDeviceDescriptiveTable[index].LQ.linkqualityAtRemoteDevice, LQI_DISP_OFFSET), FALSE);
				SetCresnetAnalogSubscriptionData (ANALOG_JOIN_FIRST_RFDEV_LINK_STATE+g_pRFDeviceDescriptiveTable[index].IDInfo.rfid, pdst->linkstate, FALSE);
		#ifdef	RF_LINK_HISTORY_LOG
				RFDesc_LogLinkTransition (index, LinkState);
		#endif
			}
		#endif
		}	else	{
			if ((LinkState == RF_DEVICE_LINK_STATE_FAIL)  || 
				(LinkState == RF_DEVICE_LINK_STATE_SLEEP) ||
				(LinkState == RF_DEVICE_LINK_STATE_DISCONNECTED) || 
				(LinkState == RF_DEVICE_LINK_STATE_POWERDOWN))	{					
				RFTransmit_ResetTxBuffers (index);
				pdst->resultcodeavailable = FALSE;			
				pdst->waitingforACK = FALSE;
			}
		}
	}
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function checks if the device is in acquire mode 	   
 *
 * \detail		  
 *
 * \date        10/11/2006
 *
 * \param	    None		
 *
 * \return		BOOL 	 				
 *
 * \retval		TRUE/FALSE	 	  
 */	
BOOL RFDesc_IsLocalDeviceInAcquire (void)	{
	return (AcquireMode);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function sets the UID of the given descriptive entry	   	     			           
 *
 * \detail		
 *						
 * \date        11/28/2006
 *					
 * \param	    UINT8 index, UINT8 *puid
 *
 * \return		None					
 *
 * \retval		None	
 */
void RFDesc_SetUID (UINT8 index, UINT8 *puid)	{
    pr_info("%s - &g_pRFDeviceDescriptiveTable[%i] = %p - puid = %p\n",
            __FUNCTION__, index, &g_pRFDeviceDescriptiveTable[index], puid);
	memcpy (&g_pRFDeviceDescriptiveTable[index].IDInfo.uid, puid, UID_MAX_SIZE);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function sets the panid of the given descriptive entry	   	     			           
 *
 * \detail		
 *						
 * \date        11/28/2006
 *					
 * \param	    UINT8 index, UINT16 panid
 *
 * \return		None					
 *
 * \retval		None	
 */
void RFDesc_SetPanID (UINT8 index, UINT16 panid)	{	
	g_pRFDeviceDescriptiveTable[index].IDInfo.panid = panid;	
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function reads the panid of the given descriptive entry	   	     			           
 *
 * \detail		
 *						
 * \date        11/28/2006
 *					
 * \param	    UINT8 index
 *
 * \return		UINT16 panid					
 *
 * \retval		0..0xFFFF
 */
UINT16	RFDesc_GetPanID (UINT8 index)	{	
	return(g_pRFDeviceDescriptiveTable[index].IDInfo.panid);		
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function sets the addrmode of the given descriptive entry	   	     			           
 *
 * \detail		
 *						
 * \date        11/28/2006
 *					
 * \param	    UINT8 index, UINT8 addrmode 
 *
 * \return		None					
 *
 * \retval		None	
 */
void RFDesc_SetAddrMode (UINT8 index, UINT8 addrmode)	{
	g_pRFDeviceDescriptiveTable[index].IDInfo.addrmode = addrmode;	
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function sets the rfid of the given descriptive entry	   	     			           
 *
 * \detail		
 *						
 * \date        11/28/2006
 *					
 * \param	    UINT8 index, UINT8 rfid	
 *
 * \return		None					
 *
 * \retval		None	
 */
void RFDesc_SetRFID (UINT8 index, UINT8 rfid)	{	
	g_pRFDeviceDescriptiveTable[index].IDInfo.rfid = rfid;	
}

UINT8	RFDesc_MaxSupportedDescriptiveEntries (void)	{		
	return (g_MaxDescriptiveTableEntries);		
}

TYPE_RF_DESCRIPTIVE_TABLE *RFDesc_GetDescriptiveEntryIDInfo (UINT8 index)	{
	//UINT16 res;
	TYPE_RF_DESCRIPTIVE_TABLE *pdesc = NULL;

	if (index < g_MaxDescriptiveTableEntries)	{
		pdesc = &g_pRFDeviceDescriptiveTable[index];				
	}	
	
	return (pdesc);			
}

INT8 *ConvertLinkStatetoString (UINT8 linkstate, INT8 *pbuf)	{
	switch (linkstate)	{
		case	RF_DEVICE_LINK_STATE_EMPTY:
			sprintf (pbuf, "Empty"); 
			break;	
		case	RF_DEVICE_LINK_STATE_TEMP_CONN_ACCESS:
			sprintf (pbuf, "TempConn"); 
			break;			
		case	RF_DEVICE_LINK_STATE_ACTIVE:
			sprintf (pbuf, "Active"); 
			break;			
		case	RF_DEVICE_LINK_STATE_FAIL:
			sprintf (pbuf, "Fail");
			break;	
		case	RF_DEVICE_LINK_STATE_SLEEP:
			sprintf (pbuf, "Sleep");
			break;			
		case	RF_DEVICE_LINK_STATE_DISCONNECTED:
			sprintf (pbuf, "Disconnected");
			break;	
		case	RF_DEVICE_LINK_STATE_POWERDOWN:
			sprintf (pbuf, "PowerDown");			
			break;	
		case	RF_DEVICE_LINK_STATE_CONNECTING:
			sprintf (pbuf, "Connecting");			
			break;	
		default:	
			sprintf (pbuf, "Undef %02x", linkstate);			
			break;	
	}	
	return (pbuf);			
}

INT8 *ConvertUIDtoString (UINT8 *puid, INT8 *pstr, UINT8 addrmode) {
	switch (addrmode)	{
		case	gAddrModeLong_c:
			sprintf (pstr, "%02x%02x%02x%02x%02x%02x%02x%02x", *(puid+0),*(puid+1),*(puid+2),*(puid+3),*(puid+4),*(puid+5),*(puid+6),*(puid+7)); 	
			break;
		case	gAddrModeShort_c:
			sprintf (pstr, "%02x%02x", *(puid+0),*(puid+1)); 	
			break;
		case	gAddrModeNone_c:
		default:		
			sprintf (pstr, ""); 	
			break;
	}
	return (pstr);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function updates the link quality received from remote device   			           
 *
 * \detail		
 * \date        09/06/2006
 * \param	    index, link quality 
 *
 * \return		result 			
 *
 * \retval		SUCCESS/FAILURE		
 */
UINT16 RFDesc_UpdateLinkQualityOfRemoteDevice (UINT8 index, UINT8 linkquality)	{
	UINT16	res = FAILURE;

	if (index < g_MaxDescriptiveTableEntries)	{
		g_pRFDeviceDescriptiveTable[index].LQ.linkqualityAtRemoteDevice = linkquality;

#ifdef	RF_NETWORK_MASTER_DEVICE		
		if (index < g_MaxSupportedRFDevices)	{
			SetCresnetAnalogSubscriptionData (ANALOG_JOIN_FIRST_RFDEV_LINK_QUALITY_AT_DEV+g_pRFDeviceDescriptiveTable[index].IDInfo.rfid, RFReceive_ConvertLinkQualityToIEEEWord(linkquality, LQI_DISP_OFFSET), FALSE);
		}
#endif

		res = SUCCESS;
	}

	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function updates the link quality on packets received from gateway at rf device   			           
 *
 * \detail		
 * \date        09/06/2006
 * \param	    index, link quality 
 *
 * \return		result 			
 *
 * \retval		SUCCESS/FAILURE		
 */
UINT16 RFDesc_UpdateLinkQualityOfLocalDevice (UINT8 index, UINT8 linkquality)	{
	UINT16	res = FAILURE;

	if (index < g_MaxDescriptiveTableEntries)	{
		g_pRFDeviceDescriptiveTable[index].LQ.linkqualityAtLocalDevice = linkquality;

#ifdef	RF_NETWORK_MASTER_DEVICE		
		if (index < g_MaxSupportedRFDevices)	{
			SetCresnetAnalogSubscriptionData (ANALOG_JOIN_FIRST_RFDEV_LINK_QUALITY_AT_HPRFGW+g_pRFDeviceDescriptiveTable[index].IDInfo.rfid, RFReceive_ConvertLinkQualityToIEEEWord(linkquality, LQI_DISP_OFFSET), FALSE);
		}
#endif

		res = SUCCESS;
	}

	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function gets local device link quality at given index	   			           
 *
 * \detail		
 * \date        01/12/2007
 *
 * \param	    index, *plinkquality
 *
 * \return		result 			
 *
 * \retval		SUCCESS/FAILURE		
 */
UINT16 RFDesc_GetLinkQualityOfLocalDevice (UINT8 index, UINT8 *plinkquality)	{	
	UINT16	res = FAILURE;

#ifdef FREESCALE_802_15_4_STACK
	*plinkquality = 0;
#else
	*plinkquality = 255;	//	255 translates to the lowest LQI value
#endif

	if (index < g_MaxDescriptiveTableEntries)	{
		*plinkquality = g_pRFDeviceDescriptiveTable[index].LQ.linkqualityAtLocalDevice;
		res = SUCCESS;
	}

	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function gets remote device link quality at given index	   			           
 *
 * \detail		
 * \date        01/12/2007
 *
 * \param	    index, *plinkquality
 *
 * \return		result 			
 *
 * \retval		SUCCESS/FAILURE		
 */
UINT16 RFDesc_GetLinkQualityOfRemoteDevice (UINT8 index, UINT8 *plinkquality)	{	
	UINT16	res = FAILURE;

#ifdef FREESCALE_802_15_4_STACK
	*plinkquality = 0;
#else
	*plinkquality = 255;	//	255 translates to the lowest LQI value
#endif	

	if (index < g_MaxDescriptiveTableEntries)	{
		*plinkquality = g_pRFDeviceDescriptiveTable[index].LQ.linkqualityAtRemoteDevice;
		res = SUCCESS;
	}

	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function gets remote device link quality of given rf device 	   			           
 *
 * \detail		
 * \date        01/12/2007
 *
 * \param	    rfid, *plinkquality
 *
 * \return		UINT16 res  			
 *
 * \retval		SUCCESS/FAILURE		
 */
UINT16 RFDesc_GetLinkQualityOfRemoteDeviceByRFID (UINT8 rfid, UINT8 *plinkquality)	{	
	UINT16	res = FAILURE;
	UINT8   index[MAX_DESCRIPTIVE_TABLE_ENTRIES];  

#ifdef FREESCALE_802_15_4_STACK
	*plinkquality = 0;
#else
	*plinkquality = 255;	//	255 translates to the lowest LQI value
#endif	

	if (RFDesc_GetActiveRFDeviceTableIndexByRFID (rfid, index))	{		
		res = RFDesc_GetLinkQualityOfRemoteDevice (index[0], plinkquality);		
	}

	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function gets local device link quality at given index	   			           
 *
 * \detail		
 * \date        03/21/2007
 *
 * \param	    UINT8 rfid, UINT8 *plinkquality
 *
 * \return		UINT16 result 			
 *
 * \retval		SUCCESS/FAILURE		
 */
UINT16 RFDesc_GetLinkQualityOfLocalDeviceByRFID (UINT8 rfid, UINT8 *plinkquality)	{	
	UINT16	res = FAILURE;
	UINT8   index[MAX_DESCRIPTIVE_TABLE_ENTRIES];  

#ifdef FREESCALE_802_15_4_STACK
	*plinkquality = 0;
#else
	*plinkquality = 255;	//	255 translates to the lowest LQI value
#endif

	if (RFDesc_GetActiveRFDeviceTableIndexByRFID (rfid, index))	{
		res = RFDesc_GetLinkQualityOfLocalDevice (index[0], plinkquality);		
	}

	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function called from device handler after a new active link has been
 *				established		   			           
 *
 * \detail		
 * \date        05/11/2007
 *
 * \param	    UINT8 index
 *
 * \return		UINT16 result 			
 *
 * \retval		SUCCESS/FAILURE		
 */
UINT16	RFDesc_SetSlaveDeviceLinkEstablishedParameters(UINT8 index)	{
	UINT16 res = FAILURE;

	if (index < g_MaxSupportedRFDevices)	{
		g_pRFDeviceDescriptiveTable[index].linkstatecnt += 1;
		g_pRFDeviceDescriptiveTable[index].linkeactiveestablishedtime = OS_RetrieveClock();
		res = SUCCESS;
	}

	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function called from device handler after a new active link has been
 *				established		   			           
 *
 * \detail		
 * \date        05/11/2007
 *
 * \param	    UINT8 index
 *
 * \return		UINT16 result 			
 *
 * \retval		SUCCESS/FAILURE		
 */
UINT16 RFDesc_GetSlaveDeviceInfoField (UINT8 index, UINT8 field, INT8 *pbuf, UINT16 *plen, UINT16 *pcrc)	{
	UINT16 res = FAILURE;
	*plen = 0;

	if ((index < g_MaxSupportedRFDevices) && (field < RF_DEVICE_INFO_MAX_DATA_FIELDS))	{
		*plen = strlen(g_pRFDeviceDescriptiveTable[index].DevField.field[field].data);	
		memcpy (pbuf, g_pRFDeviceDescriptiveTable[index].DevField.field[field].data, *plen);		
		*pcrc = g_pRFDeviceDescriptiveTable[index].DevField.field[field].crc;		
		res = SUCCESS;
	}
	*(pbuf+(*plen)) = 0;

	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function saves the device field info and crc into the descriptive table 
 *
 * \detail		
 * \date        09/10/2007
 *
 * \param	    UINT8 index, UINT8 field, UINT8 *pbuf, UINT16 len, UINT16 crc
 *
 * \return		UINT16 result 			
 *
 * \retval		SUCCESS/FAILURE		
 */
UINT16 RFDesc_SetSlaveDeviceInfoField (UINT8 index, UINT8 field, INT8 *pbuf, UINT16 len, UINT16 crc)	{
	UINT16 res = FAILURE;

	if ((index < g_MaxSupportedRFDevices) && (field < RF_DEVICE_INFO_MAX_DATA_FIELDS) && (len<(RF_DEVICE_DATA_FIELD_MAX_SIZE-1)))	{
		memcpy(g_pRFDeviceDescriptiveTable[index].DevField.field[field].data, pbuf, len);
		g_pRFDeviceDescriptiveTable[index].DevField.field[field].data[len] = 0;
		g_pRFDeviceDescriptiveTable[index].DevField.field[field].crc = crc;
		res = SUCCESS;
	}

	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function sorts the passed on index table elements by RFID  
 *
 * \detail		
 * \date        08/13/2007
 *
 * \param	    UINT8 *pindextbl, UINT8 count 
 *
 * \return		UINT16 result 			
 *
 * \retval		SUCCESS/FAILURE		
 */
UINT16 RFDesc_SortIndexElementsByRFID (UINT8* pindextbl, UINT8 count)	{	
	UINT32	i, j, item;

	if (count > 1)	{
		for (i=0; i<count; i++)	{
			for (j=i; j<count; j++)	{
				if (g_pRFDeviceDescriptiveTable[*(pindextbl+i)].IDInfo.rfid > g_pRFDeviceDescriptiveTable[*(pindextbl+j)].IDInfo.rfid)	{
					item = *(pindextbl+j);
					*(pindextbl+j) = *(pindextbl+i);	
					*(pindextbl+i) = item;
				}
			}
		}
	}

	return (SUCCESS);
}

#ifdef	RF_LINK_HISTORY_LOG
/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		log device link state transition    	   			           
 *
 * \detail		
 * \date        08/23/2007
 *
 * \param	    UINT8 index, UINT8 linkstate 
 *
 * \return		INT32 			
 *
 * \retval		SUCCESS/FAILURE		
 */
INT32	RFDesc_LogLinkTransition (UINT8 index, UINT8 linkstate)	{
	INT32	res = FAILURE;

	if (index < g_MaxDescriptiveTableEntries)	{
		g_pRFDeviceDescriptiveTable[index].COMLog.LinkStateHistory[g_pRFDeviceDescriptiveTable[index].COMLog.linkstatelogindex].time = OS_RetrieveClock();
		g_pRFDeviceDescriptiveTable[index].COMLog.LinkStateHistory[g_pRFDeviceDescriptiveTable[index].COMLog.linkstatelogindex].state = linkstate;
	
		g_pRFDeviceDescriptiveTable[index].COMLog.linkstatelogindex += 1;
		if (g_pRFDeviceDescriptiveTable[index].COMLog.linkstatelogindex >= MAX_LINK_TRANSITION_HISTORY_LOG)	{
			g_pRFDeviceDescriptiveTable[index].COMLog.linkstatelogindex = 0;
		}

		res = SUCCESS;
	}

	return	(res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		get index to the last log entry of link state transition    	   			           
 *
 * \detail		
 * \date        08/23/2007
 *
 * \param	    UINT8 index, UINT8 *pindex 
 *
 * \return		INT32 			
 *
 * \retval		SUCCESS/FAILURE		
 */
INT32	RFDesc_GetIndexofCurrentLinkTransition (UINT8 index, UINT8 *plinkstatetransition, UINT8 *plinkstatemaxentries)	{	
	INT32	res = FAILURE;
	
	*plinkstatetransition = 0;
	if (index < g_MaxDescriptiveTableEntries)	{
		if (g_pRFDeviceDescriptiveTable[index].COMLog.linkstatelogindex)	{
			*plinkstatetransition = g_pRFDeviceDescriptiveTable[index].COMLog.linkstatelogindex;
		}	else	{
			*plinkstatetransition = MAX_LINK_TRANSITION_HISTORY_LOG;
		}
		*plinkstatetransition -= 1;	
		*plinkstatemaxentries  =  MAX_LINK_TRANSITION_HISTORY_LOG;
		res = SUCCESS;
	}

	return	(res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		get link transition log at given index    	   			           
 *
 * \detail		
 * \date        08/23/2007
 *
 * \param	    UINT8 index, UINT8 *plinkstateindex, TYPE_RF_DEVICE_LINK_HISTORY *plk 
 *
 * \return		INT32 			
 *
 * \retval		SUCCESS/FAILURE		
 */
INT32	RFDesc_GetLinkTransitionLogAtIndex (UINT8 index, UINT8 *plinkstateindex, TYPE_RF_DEVICE_LINK_HISTORY **plk)	{	
	INT32	res = FAILURE;

	if ((index < g_MaxDescriptiveTableEntries) && (*plinkstateindex < MAX_LINK_TRANSITION_HISTORY_LOG))	{
		*plk = (TYPE_RF_DEVICE_LINK_HISTORY*)&g_pRFDeviceDescriptiveTable[index].COMLog.LinkStateHistory[*plinkstateindex];

		if (*plinkstateindex)	{
			*plinkstateindex -= 1;
		}	else	{
			*plinkstateindex = MAX_LINK_TRANSITION_HISTORY_LOG-1;
		}
		
		res = SUCCESS;
	}

	return	(res);
}
#endif

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function checks whether the device is defined in broadcast device list     	   			           
 *
 * \detail		
 * \date        09/14/2007
 *
 * \param	    UINT8 *index, UINT8 *puid, UINT16 panid 
 *
 * \return		INT32 			
 *
 * \retval		SUCCESS/FAILURE		
 */
INT32	RFDesc_FindBroadcastDeviceUID (UINT8 *index, UINT8 *puid, UINT16 panid)	{
	INT32	res = FAILURE;
	UINT32	i;

	//	check if device is in the list, return index 
	for (i=0; i<MAX_RF_NETWORK_SUPPORTED_DEVICES; i++)	{
		if ((memcmp(BDR.rx[i].uid, puid, UID_MAX_SIZE) == 0) && (BDR.rx[i].panid == panid))	{
			*index = i;
			res = SUCCESS;
			break;
		}
	}

	return (res);
}
#ifdef	RF_NETWORK_MASTER_DEVICE
/**
 *
 * \author      John Cheng
 *
 * \brief		This function will delete device in broadcast device list     	   			           
 *
 * \detail		
 * \date        12/14/2008
 *
 * \param	    UINT8 *puid, UINT16 panid 
 *
 * \return		INT32 			
 *
 * \retval		SUCCESS/FAILURE		
 */
INT32	RFDesc_DeleteBroadcastDeviceUID (UINT8 *puid, UINT16 panid)	
{
	INT32	res = FAILURE;
	UINT32	i;

	//	check if device is in the list 
	for (i=0; i<MAX_RF_NETWORK_SUPPORTED_DEVICES; i++)	
    {
		if ((memcmp(BDR.rx[i].uid, puid, UID_MAX_SIZE) == 0) && (BDR.rx[i].panid == panid))	
        {
            memset(BDR.rx[i].uid, 0, UID_MAX_SIZE);
			BDR.rx[i].panid = 0;
			res = SUCCESS;
			break;
		}
	}
	return (res);
}
#endif
/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		Add broadcasting device UID, panid to the list     	   			           
 *
 * \detail		
 * \date        08/23/2007
 *
 * \param	    UINT8 *puid, UINT16 panid, UINT8 seqno 
 *
 * \return		INT32 			
 *
 * \retval		SUCCESS/FAILURE		
 */
INT32	RFDesc_AddBroadcastDeviceUID (UINT8 *pindex, UINT8 *puid, UINT16 panid, UINT8 seqno)	{
	INT32	res = FAILURE;
	UINT8	t[UID_MAX_SIZE];
	UINT32	i;

	memset (t, 0, UID_MAX_SIZE);
	
	//	find an empty slot
	for (i=0; i<MAX_RF_NETWORK_SUPPORTED_DEVICES; i++)	{
		if (memcmp(BDR.rx[i].uid, t, UID_MAX_SIZE) == 0)	{
			memcpy (BDR.rx[i].uid, puid, UID_MAX_SIZE);		
			BDR.rx[i].panid    = panid;
			BDR.rx[i].sequence = seqno;
			*pindex = i;
			res = SUCCESS;
			break;
		}
	}

	if (res != SUCCESS)	{
		//	overwrite the last entry in the table
		memcpy (BDR.rx[MAX_RF_NETWORK_SUPPORTED_DEVICES-1].uid, puid, UID_MAX_SIZE);		
		BDR.rx[MAX_RF_NETWORK_SUPPORTED_DEVICES-1].panid    = panid;
		BDR.rx[MAX_RF_NETWORK_SUPPORTED_DEVICES-1].sequence = seqno;
		*pindex = MAX_RF_NETWORK_SUPPORTED_DEVICES-1;
	}

	return (res);
}


/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		Add invalid packet to log      	   			           
 *
 * \detail		
 * \date        09/18/2007
 *
 * \param	    UINT8 index, mcpsToNwkMessage_t *pMsgIn
 *
 * \return		INT32 			
 *
 * \retval		SUCCESS/FAILURE		
 */
INT32	RFDesc_LogErrorPkt (UINT8 index, mcpsToNwkMessage_t *pMsgIn)	{
#ifdef	RF_INVALID_RX_PACKET_LOG
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable+index;
			
	memcpy ((UINT8*)&pdst->COMLog.InvalidPktLog.packet[pdst->COMLog.InvalidPktLog.packetindex].hdr,  (UINT8*)&pMsgIn->msgData.dataInd,    sizeof(mcpsDataInd_t));
	memcpy ((UINT8*)&pdst->COMLog.InvalidPktLog.packet[pdst->COMLog.InvalidPktLog.packetindex].data, pMsgIn->msgData.dataInd.msdu,   pMsgIn->msgData.dataInd.msduLength);
	pdst->COMLog.InvalidPktLog.packet[pdst->COMLog.InvalidPktLog.packetindex].datalen = pMsgIn->msgData.dataInd.msduLength;

	pdst->COMLog.InvalidPktLog.packetindex += 1;
	if (pdst->COMLog.InvalidPktLog.packetindex >=  MAX_INVALID_RX_PACKET_LOG)	{	pdst->COMLog.InvalidPktLog.packetindex = 0;	}
#endif
	return (SUCCESS);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		Add invalid packet to log      	   			           
 *
 * \detail		
 * \date        09/18/2007
 *
 * \param	    UINT8 index, mcpsToNwkMessage_t *pMsgIn, UINT8 len, UINT8 *pdata
 *
 * \return		INT32 			
 *
 * \retval		SUCCESS/FAILURE		
 */
INT32	RFDesc_GetErrorPkt (UINT8 descrindex, mcpsDataInd_t **phdr, UINT8 *len, UINT8 **pdata, UINT8 *pmsgindex)	{
	INT32	res = FAILURE;

#ifdef	RF_INVALID_RX_PACKET_LOG
	TYPE_RF_DESCRIPTIVE_TABLE *pdst;	
	if (descrindex < g_MaxDescriptiveTableEntries)	{
		pdst	= g_pRFDeviceDescriptiveTable+descrindex;
		*phdr   = (mcpsDataInd_t*)&pdst->COMLog.InvalidPktLog.packet[*pmsgindex].hdr;
		*pdata	= pdst->COMLog.InvalidPktLog.packet[*pmsgindex].data;
		*len    = pdst->COMLog.InvalidPktLog.packet[*pmsgindex].datalen;

		if	(*pmsgindex)	{	*pmsgindex -= 1;	}	else	{	*pmsgindex = (MAX_INVALID_RX_PACKET_LOG-1);	}
		res = SUCCESS;
	}
#endif
	return (res);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		Add invalid packet to log      	   			           
 *
 * \detail		
 * \date        09/18/2007
 *
 * \param	    UINT8 index, UINT8 *pmsgindex, UINT8 *pmsgentries, UINT8 *ppktcnt
 *
 * \return		INT32 			
 *
 * \retval		SUCCESS/FAILURE		
 */
INT32	RFDesc_GetErrorPktIndex (UINT8 index, UINT8 *pmsgindex, UINT8 *pmsgentries, UINT8 *ppktcnt)	{
	INT32	res = FAILURE;
#ifdef	RF_INVALID_RX_PACKET_LOG
	TYPE_RF_DESCRIPTIVE_TABLE *pdst;	


	if (index < g_MaxDescriptiveTableEntries)	{
		pdst = g_pRFDeviceDescriptiveTable+index;
		*pmsgindex	  = pdst->COMLog.InvalidPktLog.packetindex;
		*pmsgentries  = MAX_INVALID_RX_PACKET_LOG; 
		if (*pmsgindex)	{	*pmsgindex -=1;	}	else	{	*pmsgindex = MAX_INVALID_RX_PACKET_LOG-1;	}
		*ppktcnt = pdst->COMLog.InvalidPktLog.packetcnt;	 
		res = SUCCESS;
	}

#endif
	return (res);
}


/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		release descriptive table entry if there are no messages in pipe       	   			           
 *
 * \detail		
 * \date        09/18/2007
 *
 * \param	    UINT8 index, UINT8 *pmsgindex, UINT8 *pmsgentries
 *
 * \return		INT32 			
 *
 * \retval		SUCCESS/FAILURE		
 */
INT32	RFDesc_ReleaseDescriptiveEntryIfPipeIsEmpty (UINT8 index)	{	
	INT32 res = FAILURE;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable+index;
	UNSIGNED pipesize, avaliable, messages;	

	if (index < g_MaxDescriptiveTableEntries)	{
		//	if there are more messages in the pipe, do not delete entry  
		RFTransmit_GetPipeAvailableMemorySizeByIndex (index, &pipesize, &avaliable, &messages);
		if (messages == 0)	{
			RFDesc_LinkStateTransition (index, RF_DEVICE_LINK_STATE_EMPTY);
			res = SUCCESS;
		}
	}

	return (res);
}
